<?php

declare (strict_types=1);
/*
 * This file is part of the Monolog package.
 *
 * (c) Jordi Boggiano <j.boggiano@seld.be>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */
namespace SimpleCalendar\plugin_deps\Monolog\Handler;

use SimpleCalendar\plugin_deps\Monolog\Formatter\FormatterInterface;
use SimpleCalendar\plugin_deps\Monolog\Formatter\NormalizerFormatter;
use SimpleCalendar\plugin_deps\Monolog\Logger;
/**
 * Handler sending logs to Zend Monitor
 *
 * @author  Christian Bergau <cbergau86@gmail.com>
 * @author  Jason Davis <happydude@jasondavis.net>
 */
class ZendMonitorHandler extends \SimpleCalendar\plugin_deps\Monolog\Handler\AbstractProcessingHandler
{
    /**
     * Monolog level / ZendMonitor Custom Event priority map
     *
     * @var array
     */
    protected $levelMap = [];
    /**
     * @param  string|int                $level  The minimum logging level at which this handler will be triggered.
     * @param  bool                      $bubble Whether the messages that are handled can bubble up the stack or not.
     * @throws MissingExtensionException
     */
    public function __construct($level = Logger::DEBUG, bool $bubble = \true)
    {
        if (!\function_exists('SimpleCalendar\\plugin_deps\\zend_monitor_custom_event')) {
            throw new \SimpleCalendar\plugin_deps\Monolog\Handler\MissingExtensionException('You must have Zend Server installed with Zend Monitor enabled in order to use this handler');
        }
        //zend monitor constants are not defined if zend monitor is not enabled.
        $this->levelMap = [Logger::DEBUG => \SimpleCalendar\plugin_deps\ZEND_MONITOR_EVENT_SEVERITY_INFO, Logger::INFO => \SimpleCalendar\plugin_deps\ZEND_MONITOR_EVENT_SEVERITY_INFO, Logger::NOTICE => \SimpleCalendar\plugin_deps\ZEND_MONITOR_EVENT_SEVERITY_INFO, Logger::WARNING => \SimpleCalendar\plugin_deps\ZEND_MONITOR_EVENT_SEVERITY_WARNING, Logger::ERROR => \SimpleCalendar\plugin_deps\ZEND_MONITOR_EVENT_SEVERITY_ERROR, Logger::CRITICAL => \SimpleCalendar\plugin_deps\ZEND_MONITOR_EVENT_SEVERITY_ERROR, Logger::ALERT => \SimpleCalendar\plugin_deps\ZEND_MONITOR_EVENT_SEVERITY_ERROR, Logger::EMERGENCY => \SimpleCalendar\plugin_deps\ZEND_MONITOR_EVENT_SEVERITY_ERROR];
        parent::__construct($level, $bubble);
    }
    /**
     * {@inheritdoc}
     */
    protected function write(array $record) : void
    {
        $this->writeZendMonitorCustomEvent(Logger::getLevelName($record['level']), $record['message'], $record['formatted'], $this->levelMap[$record['level']]);
    }
    /**
     * Write to Zend Monitor Events
     * @param string $type      Text displayed in "Class Name (custom)" field
     * @param string $message   Text displayed in "Error String"
     * @param array  $formatted Displayed in Custom Variables tab
     * @param int    $severity  Set the event severity level (-1,0,1)
     */
    protected function writeZendMonitorCustomEvent(string $type, string $message, array $formatted, int $severity) : void
    {
        zend_monitor_custom_event($type, $message, $formatted, $severity);
    }
    /**
     * {@inheritdoc}
     */
    public function getDefaultFormatter() : FormatterInterface
    {
        return new NormalizerFormatter();
    }
    public function getLevelMap() : array
    {
        return $this->levelMap;
    }
}
